/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#ifndef RF_CHANNEL_H
#define RF_CHANNEL_H

#include <QTimer>
#include "config_handler.h"
#include "serial_v3.h"
#include "statuscheck.h"
#include "miscellaneous.h"


class RF_Channel : public QObject
{
	Q_OBJECT

public:
	~RF_Channel();

	Serial *SGX_port;

	RF_Channel(int intrasys_num, int channel_num, QString portname);

	/* Return parameter value functions */
	//CW
	quint64	Runtime_SGx();
	bool RF_Enable();
	double Temperature();
	double Power_watt();
	double Power_dBm();
	double SGx_Power_dBm();
	double VGA_Attenuation();
	double IQMod_Magnitude();
	double Frequency();
	double Offset_dB();
	double Phase();
	double AutoPhase();
	bool AutoPhase_Enable();

	double Phase_NChannel(int subchannel);
	double Attenuation_NChannel(int subchannel);
	double Magnitude_NChannel(int subchannel);

	//PWM
	double PWM_Frequency();
	double PWM_Duty_Cycle();

	//DLL
	double DLL_Frequency_Limit_Upper();
	double DLL_Frequency_Limit_Lower();
	double DLL_Start_Frequency();
	double DLL_Step_Frequency();
	double DLL_Threshold();
	double DLL_Main_Delay();

	//ALL
	double ALL_Frequency_Limit_Upper();
	double ALL_Frequency_Limit_Lower();
	double ALL_Threshold();

	//DVGA
	double DVGA_Attenuation_Forward();
	double DVGA_Attenuation_Reflected();

	//SWP
	double SWP_Start_Frequency();
	double SWP_Stop_Frequency();
	double SWP_Step_Frequency();
	double SWP_Power_dBm();
	double SWP_Power_watt();
	QString SWP_Raw_Data();

	//PID
	double PID_Kp();
	double PID_Ki();
	double PID_Kd();
	double PID_Offset();
	double PID_Setpoint();
	double PID_Scaling();

	//PSU
	double PSU_Voltage_Setpoint();
	double PSU_Current_Limit();

private:
	bool channel_working = false;
	bool channel_init_part2_finished = false;
	bool channel_init_reset_finished = false;

	ConfigHandler *config;
	StatusCheck *statuschecker;

	/*
	 * Explanation: In a 16 channel system, consisting of 4 Subsystems with 4 channels each. internally each sub-system knows only 4 channels (the scope)
	 * Yet, the actual channels of each system could be number 1-4, 5-8, 9-12, 13-16, etc.
	 * The intra_system_number represents how the control system communicates internally.
	 * The channel_number is the value used for communication with the SGx_board, which could be any channel number.
	 */
	int intra_system_number;	//The number of the channel within the scope of RF subsystem.
	int target_channel_number;	//The channel number that the user sets / GUI expects (how the board is initialized basically).
	int channel_number;			//The actual channel number by the SGx board. (or what the system thinks it is anyway if it hasn't been updated properly)
	QString assigned_port_name; //The portname that belongs to this channel. (correlating SGx channel number on the particular port to the intra-system number)

	/* SGx board runtime */
	quint64 runtime_SGx = 0;

	/* Error status */
	quint64 error = 0;
	quint64 last_error = 0;
	bool error_clearing_enable = true;		//Automatic Error Clearing

	/* Temperature Readings */
	double	temperature_PA = 0.0;
	double	temperature_PA_average = 0.0;
	QVector<double>	temperature_PA_array;

	double	temperature_SG = 0.0;
	double	temperature_SG_average = 0.0;
	QVector<double>	temperature_SG_array;

	/* Power Control Modes */
	int last_power_control_mode = -1;
	int power_control_mode = 0;		//0 = Normal, 1 = Analog Input Mode, 2 = Feed Forward, 3 = PID
	bool autogain_enable = true;
	bool analog_input_mode_enable = false;
	bool PID_control_enable = false;

	/* Power readings */
	QTimer *timer_readings;
	QTimer *timer_reconnect;
	double PA_power_forward_dbm = 0.0;
	double PA_power_reflected_dbm = 0.0;
	double PA_s11 = 0.0;
	double PA_power_forward_watt = 0.0;
	double PA_power_reflected_watt = 0.0;
	double PA_s11_reflection = 0.0;
	QList<double> PA_power_forward_watt_nchannel;
	QList<double> PA_power_reflected_watt_nchannel;
	QList<double> PA_power_forward_dbm_nchannel;
	QList<double> PA_power_reflected_dbm_nchannel;
	QList<double> PA_s11_reflection_nchannel;
	QList<double> PA_s11_nchannel;


	/* Parameter Variables */
	//MISC
	bool RF_enable = false;
	int clock_source = 0;
	int firmware_version[4];
	QString firmware_version_string = "";	//string version
	QString board_manufacturer = "";
	QString board_model = "";
	QString board_serial_number = "";
	int	hardware_identity_number = -1;
	QString hardware_identity_string = "";
	int PA_type = -1;

	//External Triggering
	bool external_triggering_enable = false;
	int external_triggering_synchronization_enable = -1;
	int external_triggering_synchronization_delay_us = -1;
	int external_triggering_synchronization_delay_us_target = -1;

	//CW
	bool CW_enable = true;
	double VGA_attenuation = 0;
	double IQMod_magnitude = 0;		//percentage
	double SGx_power_dBm = 0;
	double power_watt = 0.1;
	double power_dbm = 20;
	double frequency_mhz = 434;
	double offset_db = 0.0;
	double phase_degrees = 0;

	//RF Splitter / NChannel
	bool use_substitute_channel_number = false;
	int substitute_channel_number = 201;
	int splitter_mode = 0;
	int splitter_channel_count = 0;
	QList<double> phase_degrees_nchannel;
	QList<double> magnitude_nchannel;
	QList<double> attenuation_nchannel;
	int RF_Splitter_channel_I2C_enable = -1;

	//Demodulator board
	double autophase_degrees = 0;
	bool autophase_enable = false;
	double phase_degrees_demodulator = 0;
	bool demodulator_enable = false;

	//PWM
	bool PWM_enable = false;
	double PWM_frequency = 1000;
	double PWM_frequency_maximum = 19800;
	double PWM_duty_cycle = 100;
	double PWM_duty_cycle_target = 5;
	double PWM_duty_cycle_minimum = 1;
	int PWM_triggering_mode = 1;
	int PWM_triggering_slave_port = 255;
	int PWM_triggering_slave_pin = 255;
	int PWM_triggering_master_port = 255;
	int PWM_triggering_master_pin = 255;

	//DLL
	bool DLL_enable = false;
	double DLL_frequency_limit_lower = 2400;
	double DLL_frequency_limit_upper = 2500;
	double DLL_start_frequency = 2450;
	double DLL_step_frequency = 1;
	double DLL_threshold = 0;
	double DLL_main_delay = 0;

	//ALL
	bool ALL_compatible = false;
	bool ALL_enable = false;
	double ALL_frequency_limit_upper = 1.16;
	double ALL_frequency_limit_lower = 0.88;
	double ALL_threshold = 0.2;

	//DVGA
	bool DVGA_compatible = false;
	double DVGA_attenuation_forward = 9;
	double DVGA_attenuation_reflected = 9;
	bool DVGA_amplifier_forward_enable = false;
	bool DVGA_amplifier_reflected_enable = false;
	bool DVGA_forward_ALL_enable_log;		//log the state of ALL in order to manage the DVGA safety behaviour
	bool DVGA_reflected_ALL_enable_log;		//log the state of ALL in order to manage the DVGA safety behaviour
	double target_DVGA_attenuation_forward = 9;
	double target_DVGA_attenuation_reflected = 9;
	bool target_DVGA_amplifier_forward_enable = false;
	bool target_DVGA_amplifier_reflected_enable = false;

	//SWP
	double SWP_frequency_start = 2400;
	double SWP_frequency_stop = 2500;
	double SWP_frequency_step = 1;
	double SWP_power_dbm = 30;
	double SWP_power_watt = 1;
	QString SWP_raw_data = "";

	//Analog Input Mode
	bool analog_input_enable = false;
	int analog_input_modulator = 0;
//	int analog_input_ADC_min = 0;				//unused
//	int analog_input_ADC_max = 0;				//unused
//	double analog_input_magnitude_min = 0;		//unused
//	double analog_input_magnitude_max = 0;		//unused
//	double analog_input_attenuation_min = 30;	//unused
//	double analog_input_attenuation_max = 30;	//unused
	double analog_input_VGA_attenuation = 31.5;

	//Amplifier Mode
#if defined(Q_OS_LINUX)
	bool GPIO_amplifier_mode_enable = false;
#endif
	bool amplifier_mode_enable = false;
	int RF_source = -1;

	//LED
#if defined(Q_OS_LINUX)
	LED_MODE last_LED_mode = LED_MODE::LED_MODE_INVALID;
#endif

	//PID
	double PID_prev_error = 0;
	double PID_kp = 0;
	double PID_ki = 0;
	double PID_kd = 0;
	double PID_setpoint = 0;
	double PID_integral = 0;
	double PID_scaling = 1;
	double PID_offset = 0;

	//PSU
	int PSU_count = 1;
	double PSU_voltage_setpoint = 0;
	double PSU_current_limit = 0;
	bool PSU_enable_combined = 0;
	double PSU_voltage_combined = 0;
	double PSU_current_combined = 0;
	QList<bool> PSU_enable;
	QList<double> PSU_voltage;
	QList<double> PSU_current;
	QList<double> PSU_power;
	double PSU_total_power;
	double PSU_power_efficiency;
	double PSU_dissipation_watt;

	/* Internal use functions */
	bool checkStringContents(QString inputString, QString startsWith, QString endsWith, QString string_notContains);
	bool Serial_Probe();

	/* Parameter Set functions */
	void Set_Channel(int chan_num);
	void Set_DLL_Settings();
	void Set_ALL_Settings();
	void Set_DVGA_Forward_Settings();
	void Set_DVGA_Reflected_Settings();
	void Set_Analog_Input_Modulator(int modulator);
	void Set_Analog_Input_Settings();
	void Set_PID_settings();

	/* Parameter get functions */
	void Get_Channel();
	void Get_Status();
	void Get_Identity();
	void Get_Hardware_Identity();
	void Get_PA_Type();
	void Get_SGx_Runtime();
	void Detect_Runtime_SGx_Reset();

	/* RF Splitter */
	void Set_RF_splitter_channels_I2C_enable(int intrasys_num, int enable_bitmask);
	void Set_RF_Splitter_Mode(int intrasys_num, int mode);
	void Get_RF_Splitter_Mode();

	int Identify_Power_Control_Mode();
	void Set_Analog_Input_Enable(bool enable);
//	void Get_Analog_Input_Enable();		//$command does not exist yet
	void Set_Autogain_Enable(bool enable);
	void Get_Autogain_Enable();
	void Set_Amplifier_Mode_Enable(bool enable);
	void Get_Amplifier_Mode_Enable();
	void Set_RF_Source(int intrasys_num, int source);
	void Get_RF_Source();
#if defined(Q_OS_LINUX)
	void Set_GPIO_Amplifier_Mode_Enable(bool enable);
#endif
	void Read_PA_Powers_interpret_readings();
	void Read_PA_Powers_calc_S11_dBm(double &FWD_string, double &RFL_string, double &S11);
	void Read_PA_Powers_calc_S11_watt(double &FWD, double &RFL, double &S11);
	void Read_PA_Powers_interpret_readings_NChannel();
	void Read_PA_Powers_calc_S11_dBm_NChannel(QList<double> &FWD_string, QList<double> &RFL_string, QList<double> &S11);
	void Read_PA_Powers_calc_S11_watt_NChannel(QList<double> &FWD, QList<double> &RFL, QList<double> &S11);

	void Get_RF_enable();
	void Get_PA_Temperature();
	void Get_SG_Temperature();
	void Get_Firmware_Version();
	void Get_Power_dBm();
	void Get_Power_watt();
	void Get_IQMod_Magnitude();
	void Get_VGA_attenuation();
	void Get_Frequency();
	void Get_Offset_dB();
	void Get_Phase();
	void Get_AutoPhase_Enable();
	void Get_AutoPhase();
	void Get_Phase_Demodulator();
	void Get_Phase_NChannel(int subchannel_num);
	void Get_Attenuation_NChannel(int subchannel_num);
	void Get_Magnitude_NChannel(int subchannel_num);

	void Get_Demodulator_Enable();
	void Get_Clock_Source();

	void Get_External_Triggering_Enable();
	void Get_External_Triggering_Synchronization_Enable();
	void Get_External_Triggering_Synchronization_Delay();

	void Get_DLL_Settings();
	void Get_DLL_enable();
	void Autocorrect_DLL_Minmax_Values();
	void Get_ALL_Settings();
	void Get_ALL_enable();
	void Get_DVGA_Forward_Settings();
	void Get_DVGA_Reflected_Settings();
	void Get_PWM_Settings();
	void Calculate_PWM_Duty_Cycle_Minimum();

	void Set_PSU_Enable(int intrasys_num, bool enable);
	void Get_PSU_Enable();
	void Get_PSU_IU_All_Reading();
	void Get_PSU_Voltage_Setpoint();
	void Get_PSU_Voltage_Reading();
	void Get_PSU_Current_Reading();
	void Get_PSU_Current_limit();
	void Calculate_PSU_Dissipation();

	/* Feature Availability detection */
	void determine_availability_amplifier_mode();
	void determine_availability_external_triggering();
	void determine_availability_PWM();
	void determine_availability_ALL();
	void determine_availability_DLL();
	void determine_availability_sweep();
	void determine_availability_GCS_controls();
	void determine_availability_MCS_controls();
	void determine_availability_PWRS_controls();
	void determine_availability_PWRSGDS_controls();
	void determine_availability_PSU_enable_controls();

	/* LED management */
#if defined (Q_OS_LINUX)
	void determine_LED_mode();
#endif

	/* Serial Error Handling */
	void Reconnect_Protocol_Handler();
	void Serial_Reconnect();


public slots:
	void Timer_Readings_Start(int intrasys_num, int ms);
	void Timer_Readings_Stop(int intrasys_num);
	void Initialize(int intrasys_num);
	void Initialize_part2();

	void Serial_Error_Handler(QSerialPort::SerialPortError error);

	/* SET / Execute functions */
	void Set_Channel_Target(int intrasys_num, int chan_num);
	void Set_RF_enable(int intrasys_num, bool enable);

	void Update_PA_Power(int intrasys_num);
	void Set_Power_Watt(int intrasys_num, double val);
	void Set_Power_dBm(int intrasys_num, double val);
	void Set_SGxPower_dBm(int intrasys_num, double sgx_power_dbm);
	void Set_VGA_attenuation(int intrasys_num, double attenuation_dB);
	void Set_IQMod_Magnitude(int intrasys_num, double magnitude);

	void Set_Frequency(int intrasys_num, double val);
	void Set_Offset_dB(int intrasys_num, double val);
	void Set_Phase(int intrasys_num, double val);
	void Set_AutoPhase_Enable(int intrasys_num, bool enable);
	void Set_AutoPhase(int intrasys_num, double val);
	void Set_Phase_NChannel(int intrasys_num, int subchannel_num, double val);
	void Set_Attenuation_NChannel(int intrasys_num, int subchannel_num, double val);
	void Set_Magnitude_NChannel(int intrasys_num, int subchannel_num, double val);

	void Set_Demodulator_Enable(int intrasys_num, bool enable);
	void Set_Clock_Source(int intrasys_num, int mode);
	void Set_PWM_Triggering(int intrasys_num, int mode);
	void Set_Error_Clearing_Enable(int intrasys_num, bool enable);
	void Execute_Error_Clear(int intrasys_num);
	void Execute_Reset_SGx(int intrasys_num);
	void Execute_Reset_Protection(int intrasys_num);

	void Set_External_Triggering_Enable_Configuration(int intrasys_num, bool enable);
	void Set_External_Triggering_Enable(int intrasys_num, bool enable);						//I guess these shouldn't be slots anymore?
	void Set_External_Triggering_Synchronization_Enable(int intrasys_num, bool enable);		//I guess these shouldn't be slots anymore?
	void Set_External_Triggering_Synchronization_Delay(int intrasys_num, int useconds);		//I guess these shouldn't be slots anymore?

	void Set_Power_Control_Mode(int intrasys_num, int mode);
#if defined(Q_OS_LINUX)
	void handler_GPIO_amplifier_mode_get(int intrasys_num, bool GPIO_amplifier_mode);
#endif

//	void Set_DLL_Settings(int intrasys_num, double frequency_lower, double frequency_upper, double frequency_start, double frequency_step, double threshold, double main_delay);
	void Set_DLL_Frequency_Lower_Limit(int intrasys_num, double val);
	void Set_DLL_Frequency_Upper_Limit(int intrasys_num, double val);
	void Set_DLL_Frequency_Start(int intrasys_num, double val);
	void Set_DLL_Frequency_Step(int intrasys_num, double val);
	void Set_DLL_Frequency_Threshold(int intrasys_num, double val);
	void Set_DLL_Frequency_Main_Delay(int intrasys_num, double val);
	void Set_DLL_enable(int intrasys_num, bool enable);
//	void Set_ALL_Settings(int intrasys_num, double frequency_lower_voltage, double frequency_upper_voltage, double threshold_voltage);
	void Set_ALL_Frequency_Lower_Limit(int intrasys_num, double val);
	void Set_ALL_Frequency_Upper_Limit(int intrasys_num, double val);
	void Set_ALL_Frequency_Threshold(int intrasys_num, double val);
	void Set_ALL_enable(int intrasys_num, bool enable);
	void Set_DVGA_Amplifier_Forward_Enable(int intrasys_num, bool enable);
	void Set_DVGA_Amplifier_Reflected_Enable(int intrasys_num, bool enable);
	void Set_DVGA_Attenuation_Forward(int intrasys_num, double val);
	void Set_DVGA_Attenuation_Reflected(int intrasys_num, double val);
	void Set_PWM_enable(int intrasys_num, bool enable);
	void Set_PWM_Duty_Cycle(int intrasys_num, int duty_cycle);
	void Set_PWM_Duty_Cycle_Target(int intrasys_num, int duty_cycle);
	void Set_PWM_Frequency(int intrasys_num, double frequency);
	void Set_SWP_Power_Watt(int intrasys_num, double val);
	void Set_SWP_Power_dBm(int intrasys_num, double val);
	void Set_SWP_Frequency_Start(int intrasys_num, double val);
	void Set_SWP_Frequency_Stop(int intrasys_num, double val);
	void Set_SWP_Frequency_Step(int intrasys_num, double val);
	void Execute_SWP_dBm(int intrasys_num);
	void Execute_SWP_dBm2(int intrasys_num, double freq_start, double freq_stop, double freq_step, double pow_dbm);

//	void Set_PID_enabled(int intrasys_num, bool enable);
	void Set_PID_Kp(int intrasys_num, double val);
	void Set_PID_Ki(int intrasys_num, double val);
	void Set_PID_Kd(int intrasys_num, double val);
	void Set_PID_Setpoint(int intrasys_num, double val);
	void Set_PID_Scaling(int intrasys_num, double val);
	void Set_PID_Offset(int intrasys_num, double val);
	void Execute_PID_Loop(int intrasys_num, double PID_process_variable);

	void Set_PSU_Voltage_Setpoint(int intrasys_num, double val);
	void Set_PSU_Current_Limit(int intrasys_num, double val);
	void Set_PSU_Enable_Safely(int intrasys_num, bool enable);


	/* Parameter polling */
	void poll_determine_availability_features();

	void Read_PA_Powers_dBm();
	void Read_PA_Powers_NChannel();
	void Read_PA_Powers_dBm_NChannel();
	void poll_RF_enable();
	void poll_power_dbm();
	void poll_VGA_attenuation();
	void poll_IQMod_magnitude();
	void poll_VGA_attenuation_nchannel();
	void poll_IQMod_magnitude_nchannel();
	void poll_frequency();
	void poll_offset_db();
	void poll_phase();
	void poll_phase_nchannel();
	void poll_autophase();
	void poll_autophase_enable();
	void poll_demodulator_phase();
	void poll_demodulator_enable();
	void poll_temperature_PA();
	void poll_temperature_SG();
	void poll_power_control_mode();
	void poll_SG_error();
	void poll_clock_source();
	void poll_CW_enable();
	void poll_ALL_settings();
	void poll_ALL_enable();
	void poll_DLL_settings();
	void poll_DLL_enable();
	void poll_PWM_settings();
	void poll_external_triggering_mode();
	void poll_DVGA_forward_settings();
	void poll_DVGA_reflected_settings();
	void poll_SWP_settings();
	void poll_PID_settings();

	void poll_PSU_enable();
	void poll_PSU_IU_All_Readings();
	void poll_PSU_voltage_setpoint();
	void poll_PSU_voltage();
	void poll_PSU_current();
	void poll_PSU_current_limit();
	void poll_PSU_dissipation();

	/* Remote Command Mode */
	void RCM_Live_serial_writeRead_to_SGx(QString message);
	void RCM_Live_serial_Sweep_handler(QString message);
	void RCM_Blind_serial_write_to_SGx(QString message);
	void RCM_Blind_serial_handleResponse_from_SGx();
//	void RCM_TCP_handler();

	/* TEST FUNCTIONS */
	void poll_TEST();
	void TEST(int intrasys_num, int count);

signals:
	void signal_notification(int intrasys_num, QString notification);
	void signal_channel_working(int intrasys_num, bool enabled);
	void signal_channel_initialization_complete(int intrasys_num, bool success);
	void signal_splitter_configuration(int intrasys_num, int splitter_mode, int splitter_channel_count);
	void signal_error_get(int intrasys_num, quint64 error, QStringList error_messages);
	void signal_reset_detected(int intrasys_num, int channel_number);
	void signal_reset_detected_runtime(int intrasys_num, int channel_number);
	void signal_temperature_PA_get(int intrasys_num, double temperature_PA);
	void signal_temperature_SG_get(int intrasys_num, double temperature_SG);
	void signal_firmware_version_get(int intrasys_num, int channel_number, QString version);
	void signal_board_identity_get(int intrasys_num, QString manufacturer, QString model, QString serialnumber);
	void signal_hardware_identity_get(int intrasys_num, int identity_number, QString identity_string);
	void signal_PA_type_get(int intrasys_num, int PA_type);
	void signal_channel_number_get(int intrasys_num, int channel_number);
	void signal_power_control_mode_get(int intrasys_num, int power_control_mode);

	void signal_clock_source_get(int intrasys_num, int mode);					//0 = standalone, 1 = master, 2 = slave, 3 = inline slave
	void signal_PWM_triggering_get(int intrasys_num, int mode);					//1 = Free-running, 2 = triggered master, 3 = triggered slave
	void signal_external_triggering_enable_get(int intrasys_num, bool enable);
	void signal_external_triggering_synchronization_enable_get(int intrasys_num, bool enable);
	void signal_external_triggering_synchronization_delay_get(int intrasys_num, int useconds);


	void signal_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_reflection);
	void signal_RF_enable_get(int intrasys_num, bool enable);
	void signal_sgx_power_dbm_get(int intrasys_num, double power_dbm);
	void signal_VGA_attenuation_get(int intrasys_num, double attenuation);
	void signal_IQMod_magnitude_get(int intrasys_num, double magnitude);
	void signal_power_get(int intrasys_num, double power_dbm, double power_watt);
	void signal_frequency_get(int intrasys_num, double frequency_mhz);
	void signal_offset_db_get(int intrasys_num, double offset_db);
	void signal_phase_get(int intrasys_num, double phase_degrees);				//phase value
	void signal_autophase_get(int intrasys_num, double phase_degrees);			//autophase target value
	void signal_autophase_enable_get(int intrasys_num, bool enable);			//autophase enable state
	void signal_demodulator_phase_get(int intrasys_num, double phase_degrees);	//demodulator phase value
	void signal_demodulator_enable_get(int intrasys_num, bool enable);			//demodulator enable state

	void signal_PA_power_readings_NChannel(int intrasys_num, int subchannel, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_reflection);
	void signal_NChannel_phase_get(int intrasys_num, int subchannel_num, double val);
	void signal_NChannel_magnitude_get(int intrasys_num, int subchannel_num, double val);
	void signal_NChannel_attenuation_get(int intrasys_num, int subchannel_num, double val);


	void signal_CW_enable_get(int intrasys_num, bool enable);
	void signal_DLL_enable_get(int intrasys_num, bool enable);
	void signal_DLL_settings_get(int intrasys_num, double frequency_lower, double frequency_upper, double frequency_start, double frequency_step, double threshold, double main_delay);
	void signal_ALL_compatible_get(int intrasys_num, bool compatibility);
	void signal_ALL_enable_get(int intrasys_num, bool enable);
	void signal_ALL_settings_get(int intrasys_num, double frequency_lower_voltage, double frequency_upper_voltage, double threshold_voltage);
	void signal_DVGA_compatible_get(int intrasys_num, bool compatibility);
	void signal_DVGA_forward_settings_get(int intrasys_num, bool amplifier_enable, double attenuation);
	void signal_DVGA_reflected_settings_get(int intrasys_num, bool amplifier_enable, double attenuation);
	void signal_PWM_settings_get(int intrasys_num, double frequency_hz, int duty_cycle);
	void signal_SWP_settings_get(int instrasys_num, double frequency_start, double frequency_stop, double frequency_step, double power_dbm, double power_watt);
	void signal_SWP_measurements_get(int intrasys_num, QString SWP_raw_data);
	void signal_PID_settings_get(int intrasys_num, double Kp, double Ki, double Kd, double setpoint, double scaling, double offset);

	void signal_PSU_readings_valid(int intrasys_num, bool valid);
	void signal_PSU_IU_reading_get(int intrasys_num, int psu_num, double voltage, double current, double power);
	void signal_PSU_power_efficiency_get(int intrasys_num, double efficiency);
	void signal_PSU_current_limit_get(int intrasys_num, double val);
	void signal_PSU_enable_combined_get(int intrasys_num, bool val);
	void signal_PSU_voltage_setpoint_get(int intrasys_num, double val);
	void signal_PSU_voltage_get(int intrasys_num, double val);		//Not really used...
	void signal_PSU_current_get(int intrasys_num, double val);		//Not really used...
	void signal_PSU_dissipation_get(int intrasys_num, double val);

	/* Feature availability detection */
	void signal_availability_amplifier_mode(int intrasys_num, bool visible, bool available);
	void signal_availability_external_triggering(int intrasys_num, bool visible, bool available);
	void signal_availability_PWM(int intrasys_num, bool visible, bool available);
	void signal_availability_ALL(int intrasys_num, bool visible, bool available);
	void signal_availability_DLL(int intrasys_num, bool visible, bool available);
	void signal_availability_sweep(int intrasys_num, bool visible, bool available);
	void signal_availability_GCS_controls(int intrasys_num, bool visible, bool available);
	void signal_availability_MCS_controls(int intrasys_num, bool visible, bool available);
	void signal_availability_PWRS_controls(int intrasys_num, bool visible, bool available);
	void signal_availability_PWRSGDS_controls(int intrasys_num, bool visible, bool available);
	void signal_availability_PSU_enable(int intrasys_num, bool visible, bool available);

#if defined(Q_OS_LINUX)
	/* LED mode */
	void signal_LED_mode_set(int intrasys_num, LED_MODE mode);

	/* GPIO Amplifier mode */
	void signal_GPIO_amplifier_mode_enabled_set(int intrasys_num, bool enable);
#endif

	void signal_RCM_serial_response(QString response);

};

#endif // RF_CHANNEL_H

